/* LDIR
 *
 * This program demonstrates some of the more powerfull features of the
 * SysInfo daemon. It is a light-weight version of DIR, but one that supports
 * "Long Filenames" that are stored in the FAT. SysInfo can read these
 * filenames even in an environment without Long Filename (LFN) support (such
 * as DOS or an OS/2 box).
 *
 * Usage: LDIR [pattern]
 *
 * o the default pattern is "*.*"
 * o if "pattern" does not contain wildcard characters, LDIR
 *   assumes that the names is a directory and shows all files in
 *   that directory (e.g. "LDIR C:\DOS" shows all files in the
 *   DOS directory and thus is equivalent to "LDIR C:\DOS\*.*").
 *
 * LDIR is limited, but the limitations are in the handling of special cases
 * of the DIR command. For example, in DOS the commands "DIR \BIN" and
 * "DIR \BIN\*.*" are equivalent if "BIN" is a directory. LDIR does not handle
 * this special case correctly. It blindly adds "*.*" if you do not specify
 * wildcards.
 *
 * There are no "sort" or "recurse through sub-directories" options. This is
 * intended to be an example for the use of SysInfo. More information is in
 * the manuals of SysInfo and Daemon server. SysInfo is free (but copyrighted).
 * You should find it whereever you found this LDIR utility. (In fact, you
 * already have it in this package, but you lack the manual). Daemon server
 * is not free, but an evaluation version is available (probably in the same
 * directory/forum where you got this utility from).
 *
 * I compiled this program with Borland C++ 3.1, using the following commands:
 *
 *     bcc ldir.c sysinfo&.obj dlib.obj
 *
 *     dbind ldir sysinfo.mod
 *
 * The required files the SysInfo package are included, but you need to
 * download the "Daemon server" separately for the files DAEMON.H, DLIB.OBJ
 * and DBIND.EXE.
 *
 * Copyright (C) ITB CompuPhase 1996-1999. You are permitted to copy or modify
 * this file. No warranties.
 */

#include <dir.h>
#include <dos.h>
#include <stdio.h>
#include <string.h>
#include "daemon.h"
#include "sysinfo.h"

#define SYSINFO 3       /* we know it will always be handle 3 */

static char *decode_attr(char *str, char attrib)
{
  str[0] = (attrib & FA_RDONLY) ? 'R' : '.';
  str[1] = (attrib & FA_HIDDEN) ? 'H' : '.';
  str[2] = (attrib & FA_SYSTEM) ? 'S' : '.';
  str[3] = (attrib & FA_DIREC)  ? 'D' : '.';
  str[4] = (attrib & FA_ARCH)   ? 'A' : '.';
  str[5] = '\0';
  return str;
}

static char *decode_date(char *str, unsigned date)
{
  sprintf(str, "%2d/%02d/%4d", date & 0x1f, (date >> 5) & 0x0f,
          1980 + ((date >> 9) & 0x7f));
  return str;
}

static char *decode_time(char *str, unsigned time)
{
  sprintf(str, "%2d:%02d:%02d", (time >> 11) & 0x1f, (time >> 5) & 0x3f,
          (time & 0x1f) * 2);
  return str;
}

void main(int argc, char *argv[])
{
  extern D_TABLE sysinfo_table;
  char str[260], path[260], *ptr;
  char attrstr[8], datestr[12], timestr[10];
  struct ffblk ffblk;
  int done;

  if (!d_init()) {
    printf("Daemon server not bound\n");
    return;
  } /* if */
  sprintf(str,"%s/sysinfo.mod", argv[0]);
  if (d_load(str, 0x82) > 0xff) {
    printf("Error loading SysInfo daemon\n");
    return;
  } /* if */
  d_linkall(SYSINFO, sysinfo_table);

  if (argc == 1)
    strcpy(str, "*.*");         /* default search pattern */
  else
    strcpy(str, argv[1]);

  /* Here we also fix a buggy feature in a buggy way. If you said "LDIR \TMP",
   * the parameter is expanded to "\TMP\*.*". But this is only correct if "TMP"
   * refers to a directory, not to a file.
   */
  if (strchr(str,'?')==NULL && strchr(str,'*')==NULL) {
    /* NOTE: we *know* that "str" holds a string with a length > 0, so
     * one can savely ask the character at position strlen(str)-1.
     */
    if (str[strlen(str)-1]!='\\')
      strcat(str, "\\");        /* add backslash if it is not already there */
    strcat(str, "*.*");         /* add default search pattern */
  } /* if */

  /* First get the full name of the path, so we can say "Files in
   * directory: ...". We also need the path later, because the findfirst()
   * and findnext() functions only return the filename (not the path). But
   * you need to pinpoint what precise file you mean if you want to get its
   * long filename.
   */
  strcpy(path, str);
  si_fullpath(path, path, 0);   /* resolve the name */
  if ((strchr(path,'?')!=NULL || strchr(path,'*')!=NULL) && (ptr=strrchr(path,'\\'))!=NULL) {
    if (ptr==path+2)            /* if it is the root directory, keep the "\" */
      ptr++;
    *ptr = '\0';                /* strip filename/pattern from path */
  } /* if */
  si_fullpath(path, path, 2);   /* get long path name */
  printf("Files in directory: %s\n\n", path);

  /* now do all the files */
  done = findfirst(str, &ffblk, FA_HIDDEN|FA_SYSTEM|FA_DIREC);
  while (!done) {
    if (ffblk.ff_name[0] == '.') {
      /* special case . and .. filenames */
      ptr = ffblk.ff_name;
    } else {
      /* First prepend the path before the name. Otherwise si_fullpath() may
       * not find the files, and thus not be able to find the long name.
       */
      sprintf(str, "%s\\%s", path, ffblk.ff_name);
      si_fullpath(str, str, 2);
      ptr = strrchr(str, '\\') + 1;
    } /* if */

    /* Check whether the resulting "long" name is identical to the
     * short name. If so, there is no long name.
     */
    if (strcmp(ptr,ffblk.ff_name) == 0)
      ptr = "";

    printf("%s  %s  %s  %8ld  [%-12s]  %s\n",
           decode_attr(attrstr,ffblk.ff_attrib),
           decode_date(datestr,ffblk.ff_fdate),
           decode_time(timestr,ffblk.ff_fdate),
           ffblk.ff_fsize, ffblk.ff_name,
           ptr);

    done = findnext(&ffblk);
  } /* while */

  d_unload(SYSINFO);
}

