/* SHOWINFO
 *
 * This program displays all the information that the SysInfo daemon can
 * detect.
 */

#include <stdio.h>
#include "daemon.h"
#include "sysinfo.h"

#define HIBYTE(a)       ((unsigned short)(a) >> 8)
#define LOBYTE(a)       ((unsigned short)(a) & 0xff)

void main(void)
{
  extern D_TABLE sysinfo_table;         /* from SYSINFO&.OBJ */
  short sysinfo_handle;

  if (!d_init()) {
    printf("Daemon server not present\n");
    return;
  } /* if */

  if ((sysinfo_handle = d_load("sysinfo.mod", 0)) > 0xff) {
    printf("Error %d in loading SysInfo\n", sysinfo_handle >> 8);
    return;
  } /* if */
  d_linkall(sysinfo_handle, sysinfo_table);

  /* general version information */
  {
    int daemon_ver = d_version(0);
    int si_ver = d_version(sysinfo_handle);

    printf("Daemon server version: %x.%02x\n",
           HIBYTE(daemon_ver), LOBYTE(daemon_ver));
    printf("SysInfo version:     : %x.%02x\n\n",
           HIBYTE(si_ver), LOBYTE(si_ver));
  }

  /* The detection routines.
   * To keep the number of local variables low, and at the same time give
   * descriptive names to variables, most variables are declared just before
   * they are needed. So every section of detection code starts a new nested
   * statement bock, which (at its turn) starts with all variables.
   */
  { /* *** the system */
    short cpu, fpu, bus, mode;
    char *bus_str[] = { "PC or PC-XT (8-bit)", "PC Junior",
                        "8088/8086 (unknown, 8-bit)", "80286 XT",
                        "ISA (16-bit)", "EISA", "MCA", "PCI" };
    char *fpu_str[] = { "not present", "8087", "80287", "80387", "80487",
                        "Pentium with FPU", "Pentium with \"flawed\" FPU" };

    si_system(&cpu, &fpu, &bus, &mode);
    if (cpu == 0)
      printf("CPU:       8088 or 8086\n");
    else
      printf("CPU:       80%d86\n", cpu);
    printf("FPU:       ", fpu_str[fpu & 0x0f]);
    if (fpu==FPU_NONE || (fpu & 0x0f)!=FPU_NONE)
      printf("%s", fpu_str[fpu & 0x0f]);
    if ((fpu & 0x0f)!=0 && (fpu & 0x0f)!=fpu)
      printf(" + ");            /* there are two co-processors */
    if ((fpu & 0x10) != 0)
      printf("Weitek Abacus");
    printf("\n");
    printf("CPU mode:  %s\n", (mode==0 || mode==1) ? "real" : "V86");
    printf("Bus type:  %s\n", bus_str[bus]);
  } /* *** the system */


  { /* *** PCMCIA support */
    unsigned short version;
    short sockets;

    si_pcmcia(&version, &sockets);
    printf("PCMCIA:    ");
    if (version == 0)
      printf("unavailable\n");
    else
      printf("%d sockets, version %x.%02x\n", sockets,
             HIBYTE(version), LOBYTE(version));
  } /* *** PCMCIA support */


  { /* *** the keyboard */
    short keyboard;

    keyboard = si_keyboard();
    printf("\nKeyboard:  %s\n", keyboard ? "extended (101 or 102 keys)" :
           "standard (83 or 84 keys)");
  } /* *** the keyboard */


  { /* *** the mouse */
    short type, driver, addr, port;
    short numserial;
    PORT_INFO rs232_list[RS232_NUM];
    char *mouse_str[] = { "not installed", "bus mouse", "serial mouse",
                          "inport mouse", "IBM PS/2 style mouse", "HP mouse",
                          "unknown type" };

    si_mouse(&type, &driver, &addr);
    printf("\nMouse:     %s\n", mouse_str[type]);
    if (addr != 0) {
      // find the COM port number from the RS232 base address
      numserial = si_rs232(rs232_list, RS232_NUM);
      for (port=0; port<numserial && addr!=rs232_list[port].port; port++)
        /* nothing */;
      printf("           installed on port COM%d:\n", port+1);
    } /* if */
    if (driver != 0)
      printf("           driver is version %x.%02x\n",
             HIBYTE(driver), LOBYTE(driver));
    else
      printf("           mouse driver is not installed\n");
  } /* *** the mouse */


  { /* *** memory */
    unsigned short cmos_base, cmos_ext;
    unsigned short dos_mem;
    unsigned short xms_version, xms_freemem;
    unsigned short ems_version, ems_freepages;

    si_memory(&dos_mem, &cmos_base, &cmos_ext);
    si_xms(&xms_version, &xms_freemem);
    si_ems(&ems_version, &ems_freepages);
    printf("\nMemory:    %d kBytes base memory, (BIOS indicates %d kBytes)\n"
           "           %d kBytes extended memory\n", cmos_base, dos_mem, cmos_ext);
    if (xms_version>0)
      printf("      XMS: %d kBytes free (XMS version %x.%02x)\n", xms_freemem,
             HIBYTE(xms_version), LOBYTE(xms_version));
    else
      printf("      XMS: not available\n");
    if (ems_version>0)
      printf("      EMS: %d 16-kByte pages free (EMS version %x.%02x)\n",
             ems_freepages, HIBYTE(ems_version), LOBYTE(ems_version));
    else
      printf("      EMS: not available\n");
  } /* *** memory */


  { /* *** video subsystem */
    #define MAX_MODES   30
    short type, software;
    unsigned short memory, soft_version;
    char *video_hw_str[] = {
      "unknown",                           "no video hardware or unknown",
      "MDA (monochrome text)",             "CGA",
      "EGA",                               "EGA with CGA display",
      "EGA with monochrome (MDA) display", "MCGA",
      "MCGA with monochrome display",      "VGA",
      "VGA with monochrome display",       "Hercules (with MDA display)",
      "SuperVGA",                          "SuperVGA with monochrome display",
      "XGA",                               "XGA with monochrome display"
    };
    char *video_sw_str[] = { "unknown", "IBM AI", "IBM DQMS", "VESA" };
    VIDEO_MODE modelist[MAX_MODES];
    short nummodes;
    int i;

    si_video(&type, &memory, NULL, &software, &soft_version);
    nummodes = si_modelist(modelist, MAX_MODES);
    printf("\nVideo:     %s with %d kBytes video memory\n", video_hw_str[type],
           memory);
    printf("Interface: %s, version %x.%02x\n", video_sw_str[software],
           HIBYTE(soft_version), LOBYTE(soft_version));
    printf("Summary of video modes:\n");
    for (i=0; i<MAX_MODES && i<nummodes; i++) {
      printf("\t%03x  %8s %4d x%4d  ", modelist[i].mode,
        modelist[i].graphic ? "graphic" : "text",
        modelist[i].resx, modelist[i].resy);
      if (modelist[i].colour_bits == 1)
        printf("monochrome\n");
      else if (modelist[i].colour_bits >= 24)
        printf("RGB\n");
      else
        printf("%ld colours\n", 1L << modelist[i].colour_bits);
    } /* for */
    if (nummodes > MAX_MODES)
      printf("\t... (more modes are supported than displayed here)\n");
  } /* ** video subsystem */


  { /* *** logical drives */
    short numdrives;
    DRIVE_INFO drives[DRIVE_NUM];
    char drive[10], name[128];
    char *drive_str[] = { "-", "Removeable", "Fixed", "RAM", "CD-ROM",
                          "Network", "Virtual" };
    char *ddrive_str[] = { "-", "5\" 360k", "5\" 1.2M", "3\" 720k",
                           "3\" 1.44M", "probably 3\" 2.88M",
                           "3\" 2.88M", "5\" 320k", "unknown type" };
    char *drvflags_str[] = { "case sensitive", "case preserving", "Unicode",
                             "undefined", "undefined", "undefined",
                             "long filenames", "compressed medium" };
    int i, flag, firstflag;

    numdrives = si_drives(drives, DRIVE_NUM);
    printf("\nDisk drives:\n");
    for (i=0; i<numdrives; i++) {
      printf("\t%c:  %s", drives[i].letter, drive_str[drives[i].type]);
      if (drives[i].type==DRIVE_NETWORK || drives[i].type==DRIVE_SUBST) {
        sprintf(drive, "%c:\\", drives[i].letter);
        si_fullpath(name, drive, 0);
        printf("\t%s", name);
      } else if (drives[i].type==DRIVE_REMOVEABLE) {
        printf("\t%s", ddrive_str[si_diskette(drives[i].letter)]);
      } /* if */
      if (drives[i].flags != 0) {
        firstflag = 1;
        printf("\t");
        for (flag=0; flag<8; flag++) {
          if ((drives[i].flags & (1 << flag)) != 0) {
            if (!firstflag)
              printf(", ");
            printf("%s", drvflags_str[flag]);
            firstflag = 0;
          } /* if */
        } /* for */
      } /* if */
      printf("\n");
    } /* for */
  } /* *** logical drives */


  { /* *** serial ports */
    short numserial;
    PORT_INFO rs232_list[RS232_NUM];
    char *rs232_str[] = {
      "none",                         "8250, 8250-B or 8250C",
      "16450 or 8250A",               "16C1450",
      "16550A (with defective FIFO)", "16550AF/C/CF (with FIFO)",
      "16C1550 (with FIFO)",          "16552 dual UART (with FIFO)",
      "82510 (with FIFO)"
    };
    int i;

    numserial = si_rs232(rs232_list, RS232_NUM);
    printf("\nSerial ports:\n");
    for (i=0; i<numserial; i++)
      printf("\tCOM%d:  %s\tat I/O address %x\n", i+1,
             rs232_str[rs232_list[i].type], rs232_list[i].port);
  } /* *** serial ports */


  { /* *** parallel ports */
    short numparallel;
    PORT_INFO lpt_list[LPT_NUM];
    char *lpt_str[] = {
       "none",          "unidirectional",
       "bidirectional", "ECP", "ECP/FPP",
       "network"
    };
    char name[128];
    int i;

    numparallel = si_lpt(lpt_list, LPT_NUM);
    printf("\nParallel ports:\n");
    for (i=0; i<numparallel; i++) {
      printf("\tLPT%d:  %s", i+1, lpt_str[lpt_list[i].type]);
      if (lpt_list[i].type == LPT_NET) {
        si_lptname(name, i);
        printf("\t%s", name);
      } else {
        printf("\tat I/O address %x", lpt_list[i].port);
      } /* if */
      printf("\n");
    } /* for */
  } /* *** parallel ports */


  d_unload(sysinfo_handle);
}

